/*
+---------------------------------------------------------+
| Frances Ruiz and Michael J. Ruiz                        |
+---------------------------------------------------------|
| License:                                                |
| Creative Commons Attribution-NonCommerial 4.0 Unported  |
| http://creativecommons.org/licenses/by-nc/4.0/legalcode |
+---------------------------------------------------------+
| You are free to:                                        |
|                                                         |
| Share  copy and redistribute the material in any       |
|         medium or format                                |
| Adapt  remix, transform, and build upon the material   |
|                                                         |
| The licensor cannot revoke these freedoms as long as    |
| you follow the license terms.                           |
|                                                         |
| Attribution  You must give appropriate credit, include |
| this license, and indicate if changes were made. You    |
| may do so in any reasonable manner, but not in any way  |
| that suggests the licensor endorses you or your use.    |
|                                                         |
| Sample Credit Line: Courtesy Frances and Michael Ruiz   |
|                                                         |
| NonCommercial  You may not use the material for        |
|   commercial purposes.                                  |
|                                                         |
| No additional restrictions  You may not apply legal    |
| terms or technological measures that legally restrict   |
| others from doing anything the license permits.         |
+---------------------------------------------------------+
*/

RuizKit = window.RuizKit || {};

RuizKit = function () {
	//namespace for svg
	var svgNS = "http://www.w3.org/2000/svg";

	//global vars for dealing with times
	var intervalId = null;
	var timeoutId = null;

	//Constructor for the Slider objects
	function Slider (mySVG, x, y, numTicks) {
		//Create the subObjects that make up the Slider
		this.mySVG = mySVG;
		this.bgRects = new BGRects(this.mySVG, x, y, 360, 33);
		this.leftArrow = new ArrowButton(this.mySVG, x + 7.5, y + 16.5, x + 29.5, y + 6.5, x + 29.5, y + 26.5, "L");
		this.rightArrow = new ArrowButton(this.mySVG, x + 329.5, y + 6.5, x + 351.5, y + 16.5, x + 329.5, y + 26.5, "R");
		this.rulerMarks = new RulerMarks(this.mySVG, x + 45, x + 315, y + 2, y + 10, numTicks);
		this.sliderControl = new SliderControl(this.mySVG, this.rulerMarks.tickExes[0] - 8, y - 5.5, 16, 44);

		//Set up vars that are important for the mouse events
		this.pressedArrowPoly = null;
		
		//Add event listeners to the Slider object
		this.addListeners();
		
		//the next part needed to be able to remove the mousemove event listener
		//TODO: where to declare these vars...
		var self = this;
		var xOffset = 8;
		var curX;
		var newX;
		var dx;
		var mouseX;
		this.sliderDragHandler = function () {		
			curX = parseInt(self.sliderControl.sliderControlLine.getAttribute("x1"));
			mouseX = event.clientX - xOffset; //adjust event x value to get the sliderControl coordinates
			dx = mouseX - curX;
			newX = curX + dx;
			if (newX < self.rulerMarks.tickExes[0]) {
				newX = self.rulerMarks.tickExes[0];
			} else if (newX > self.rulerMarks.tickExes[self.rulerMarks.tickExes.length-1]) {
				newX = self.rulerMarks.tickExes[self.rulerMarks.tickExes.length - 1];
			}

			self.sliderControl.sliderControl.setAttribute("x", newX - 8);
			self.sliderControl.sliderControlBG.setAttribute("x", newX - 6.5);
			self.sliderControl.sliderControlLine.setAttribute("x1", newX);
			self.sliderControl.sliderControlLine.setAttribute("x2", newX);
			
			self.onSliderDrag();
		};
	}

	//Creates the background of the slider and appends to svg object that is passed in
	//background of the slider is an object with two rectangles: a rectangle
	//with a white border and black fill with a beige-toned rectangle on top of it
	function BGRects (mySVG, x, y, width, height) {
		this.bgRectBack = document.createElementNS(svgNS, "rect");
		this.bgRectFront = document.createElementNS(svgNS, "rect");
		
		this.bgRectBack.setAttribute("x", x);
		this.bgRectBack.setAttribute("y", y);
		this.bgRectBack.setAttribute("width", width);
		this.bgRectBack.setAttribute("height", height);
		this.bgRectBack.setAttribute("fill", "#000000");
		this.bgRectBack.setAttribute("stroke", "#FFFFFF");
		this.bgRectBack.setAttribute("stroke-width", 1.5);
		mySVG.appendChild(this.bgRectBack);
			
		this.bgRectFront.setAttribute("x", x + 2.5);
		this.bgRectFront.setAttribute("y", y + 2.5);
		this.bgRectFront.setAttribute("width", width - 3.5);
		this.bgRectFront.setAttribute("height", height - 3.5);
		this.bgRectFront.setAttribute("fill", "#E6B88A");
		mySVG.appendChild(this.bgRectFront);
	}

	//Creates an arrow button and appends to svg object that is passed in
	//input values are the three x and y points for the arrow, along with
	//a direction (either "L" or "R") for direction arrow faces
	function ArrowButton (mySVG, x1, y1, x2, y2, x3, y3, direction) {
		this.arrowPoly = document.createElementNS(svgNS, "polygon");
		this.highLine = document.createElementNS(svgNS, "line");
		this.sideLine = document.createElementNS(svgNS, "line");
		this.botLine = document.createElementNS(svgNS, "line");
			
		var pointsStr = x1.toString() + "," + y1.toString() + " "
						+ x2.toString() + "," + y2.toString() + " "
						+ x3.toString() + "," + y3.toString();
			
		this.arrowPoly.setAttribute("points", pointsStr);		
		this.arrowPoly.setAttribute("fill", "#E6B88A");
		this.highLine.setAttribute("x1", x1);
		this.highLine.setAttribute("y1", y1);
		this.highLine.setAttribute("x2", x2);
		this.highLine.setAttribute("y2", y2);
		if (direction == "L") {
			this.sideLine.setAttribute("x1", x2 - 0.5);
			this.sideLine.setAttribute("y1", y2);
			this.sideLine.setAttribute("x2", x2 - 0.5);
			this.sideLine.setAttribute("y2", y3);
			this.botLine.setAttribute("y2", y3 - 0.5);
			this.botLine.setAttribute("y1", y3 - 10.5);
			this.direction = -1;
		}
		else {
			this.sideLine.setAttribute("x1", x1 - 0.5);
			this.sideLine.setAttribute("y1", y3);
			this.sideLine.setAttribute("x2", x1 - 0.5);
			this.sideLine.setAttribute("y2", y1);
			this.botLine.setAttribute("y2", y2 - 0.5);
			this.botLine.setAttribute("y1", y3 - 0.5);
			this.direction = 1;
		}
		this.botLine.setAttribute("x2", x2);
		this.botLine.setAttribute("x1", x1);
		this.highLine.setAttribute("stroke", "#FFFFFF");
		this.sideLine.setAttribute("stroke", "#AD8A66");
		this.botLine.setAttribute("stroke", "#AD8A66");
			
		mySVG.appendChild(this.arrowPoly);
		mySVG.appendChild(this.highLine);
		mySVG.appendChild(this.sideLine);
		mySVG.appendChild(this.botLine);
	}

	//RulerMarks creates tick marks, along with numeric labels
	//must be given the startX and endX of the range over which ticks will
	//be drawn, as well as y1 and y2 for vertical start and end pts, and number of ticks
	//creates an object that stores an array of x coordinates of the tick marks,
	//the tick marks themselves (svg lines), and the tick numbers (svg texts)
	function RulerMarks(mySVG, startX, endX, y1, y2, numTicks) {
		//TODO: wish it wasn't so specific for the 256...more general?
		this.tickExes = [];
		this.tickMarks = [];
		this.tickNums = [];
			
		var rulerLen = endX - startX;
		this.tickSpace = parseInt(rulerLen / numTicks);
		var line, num, myX, myXInt; //define here or elsewhere more efficient?
		
		myX = parseInt(startX + this.tickSpace / 2); //to center things
		if (numTicks != 256) {
			for (i = 0; i < numTicks; i++) {
				this.tickExes.push(myX);
				line = document.createElementNS(svgNS, "line");
				line.setAttribute("x1", myX);
				line.setAttribute("y1", y1);
				line.setAttribute("x2", myX);
				line.setAttribute("y2", y2);
				line.setAttribute("stroke", "#000000");
				num = document.createElementNS(svgNS, "text");
				num.setAttribute("x", myX);
				num.setAttribute("y", y2 + 12);
				num.textContent = i.toString();
				num.setAttribute("text-anchor", "middle");
				num.setAttribute("font-family", "sans-serif");
				num.setAttribute("font-size", 12);
				mySVG.appendChild(line);
				mySVG.appendChild(num);
				this.tickMarks.push(line);
				this.tickNums.push(line);
				myX = myX + this.tickSpace;
			}
		}
		else {
			myX = myX + 5;
			for (i = 0; i < numTicks; i++) {
				//myX = Math.floor(startX + i * this.tickSpace);
				this.tickExes.push(myX);
				if (i % 16 == 0) {
					line = document.createElementNS(svgNS, "line");
					line.setAttribute("x1", myX);
					line.setAttribute("y1", y1);
					line.setAttribute("x2", myX);
					line.setAttribute("y2", y2);
					line.setAttribute("stroke", "#000000");
					mySVG.appendChild(line);
					this.tickMarks.push(line);
				}
				if (i % 32 == 0) {
					num = document.createElementNS(svgNS, "text");
					num.setAttribute("x", myX);
					num.setAttribute("y", y2 + 12);
					num.textContent = i.toString();
					num.setAttribute("text-anchor", "middle");
					num.setAttribute("font-family", "sans-serif");
					num.setAttribute("font-size", 12);
					mySVG.appendChild(num);
					this.tickNums.push(num);
				}
				if (i == 255) {
					line = document.createElementNS(svgNS, "line");
					line.setAttribute("x1", myX);
					line.setAttribute("y1", y1);
					line.setAttribute("x2", myX);
					line.setAttribute("y2", y2);
					line.setAttribute("stroke", "#000000");
					mySVG.appendChild(line);
					this.tickMarks.push(line);
					
					num = document.createElementNS(svgNS, "text");
					num.setAttribute("x", myX);
					num.setAttribute("y", y2 + 12);
					num.textContent = (i).toString(); //not 256
					num.setAttribute("text-anchor", "middle");
					num.setAttribute("font-family", "sans-serif");
					num.setAttribute("font-size", 12);
					mySVG.appendChild(num);
					this.tickNums.push(num);
				}
				myX = myX + this.tickSpace;
			}
		}
	}

	//creates an svg slider control object that will be used
	//to move back and forth across the ruler
	function SliderControl (mySVG, x, y, width, height) {
		this.sliderControlBG = document.createElementNS(svgNS, "rect");
		this.sliderControlBG.setAttribute("x", x+1.5);
		this.sliderControlBG.setAttribute("y",y+1.5);
		this.sliderControlBG.setAttribute("width", width);
		this.sliderControlBG.setAttribute("height", height);
		this.sliderControlBG.setAttribute("fill-opacity", 0);
		//this.sliderControlBG.setAttribute("fill", "none");
		this.sliderControlBG.setAttribute("stroke", "#000000");
		this.sliderControlBG.setAttribute("stroke-width", 1.5);
		mySVG.appendChild(this.sliderControlBG);
		
		this.sliderControl = document.createElementNS(svgNS, "rect");
		this.sliderControl.setAttribute("x", x);
		this.sliderControl.setAttribute("y", y);
		this.sliderControl.setAttribute("width", width);
		this.sliderControl.setAttribute("height", height);
		this.sliderControl.setAttribute("fill-opacity", 0);
		//this.sliderControl.setAttribute("fill", "none");
		this.sliderControl.setAttribute("stroke", "#FFFFFF");
		this.sliderControl.setAttribute("stroke-width", 1.5);
		mySVG.appendChild(this.sliderControl);
			
		this.sliderControlLine = document.createElementNS(svgNS, "line");
		this.sliderControlLine.setAttribute("x1", x + 8);
		this.sliderControlLine.setAttribute("x2", x + 8);
		this.sliderControlLine.setAttribute("y1", y + 8);
		this.sliderControlLine.setAttribute("y2", y + 38);
		this.sliderControlLine.setAttribute("stroke", "#555555");
		mySVG.appendChild(this.sliderControlLine);
	}

	//arrowDown is called to update the arrow button styles to make
	//the button appear pressed down and to moveSlider once TODO
	Slider.prototype.arrowDown = function () {
		var pressedArrowButton = null;
		if (this.pressedArrowPoly === this.leftArrow.arrowPoly) {
			pressedArrowButton = this.leftArrow;
		}
		else {
			pressedArrowButton = this.rightArrow;
		}
		pressedArrowButton.highLine.setAttribute("stroke", "#AD8A66");
		pressedArrowButton.sideLine.setAttribute("stroke", "#FFFFFF");
		pressedArrowButton.botLine.setAttribute("stroke", "#FFFFFF");	
			
		//move the slider based on the pressed arrows L or R direction (-1 and 1)
		this.moveSlider(pressedArrowButton.direction);
		if (typeof(this.onArrowDown) == "function") {
			this.onArrowDown();
		}
	};

	//startInterval sets up an interval to call arrowDown repeatedly
	//when mouse is being held down on an arrow
	Slider.prototype.startInterval = function () {
		var intervalTime = 50;
		if (this.rulerMarks.tickExes.length == 256) {
			intervalTime = 5;
		}
		var self = this;
		intervalId = setInterval(function () {self.arrowDown();}, intervalTime);
	};

	//arrowDownHandler is called on mousedown arrow events
	//sets up timers for if mouse is held down for a longer time,
	//and calls the arrowDown function
	Slider.prototype.arrowDownHandler = function () {
		this.pressedArrowPoly = event.target;
		var self = this;
		this.arrowDown(self);
		timeoutId = setTimeout(function () {self.startInterval();}, 500);
	};
			
	//arrowUp is called to update the arrow
	//button styles to make the button appear unpressed
	Slider.prototype.arrowUp = function () {
		var pressedArrowButton = null;
		if (this.pressedArrowPoly === this.leftArrow.arrowPoly) {
			pressedArrowButton = this.leftArrow;
		}
		else {
			pressedArrowButton = this.rightArrow;
		}
		pressedArrowButton.highLine.setAttribute("stroke", "#FFFFFF");
		pressedArrowButton.sideLine.setAttribute("stroke", "#AD8A66");
		pressedArrowButton.botLine.setAttribute("stroke", "#AD8A66");		
	};
		
	//arrowUpHandler is called on mouseUp and mouseOut arrow events
	//clears the timers associated with the mouse being held down
	//and calls arrowUp function to update the arrow styles
	Slider.prototype.arrowUpHandler = function () {
		clearInterval(intervalId);
		clearTimeout(timeoutId);
		this.arrowUp();
		this.pressedArrowPoly = null;
		if (typeof(this.onArrowUp) == "function") {
			this.onArrowUp();
		}
	};

	Slider.prototype.addListeners = function () {
		var slider = this;
		this.leftArrow.arrowPoly.addEventListener("mousedown",
				function () {
					slider.arrowDownHandler();
				});
		this.leftArrow.arrowPoly.addEventListener("mouseup",
				function () {
					slider.arrowUpHandler();
				});
		this.leftArrow.arrowPoly.addEventListener("mouseout",
				function () {
					slider.arrowUpHandler();
				});
		this.rightArrow.arrowPoly.addEventListener("mousedown",
				function () {
					slider.arrowDownHandler();
				});
		this.rightArrow.arrowPoly.addEventListener("mouseup",
				function () {
					slider.arrowUpHandler();
				});
		this.rightArrow.arrowPoly.addEventListener("mouseout",
				function () {
					slider.arrowUpHandler();
				});
		this.sliderControl.sliderControl.addEventListener("mousedown",
			function () {
				window.addEventListener("mousemove", slider.sliderDragHandler);
				window.addEventListener("mouseup", function () {
					window.removeEventListener("mousemove", slider.sliderDragHandler);
					slider.snapToClosest();
				});
			});
		this.sliderControl.sliderControlBG.addEventListener("mousedown",
			function () {
				window.addEventListener("mousemove", slider.sliderDragHandler);
				window.addEventListener("mouseup", function () {
					window.removeEventListener("mousemove", slider.sliderDragHandler);
					slider.snapToClosest();
				});
			});
		this.sliderControl.sliderControlLine.addEventListener("mousedown",
			function () {
				window.addEventListener("mousemove", slider.sliderDragHandler);
				window.addEventListener("mouseup", function () {
					window.removeEventListener("mousemove", slider.sliderDragHandler);
					slider.snapToClosest();
				});
			});
	};

	//moveSlider moves a slider over by one step
	Slider.prototype.moveSlider = function (direction) {
		//find current x pos of slider control line
		var curX = this.sliderControl.sliderControlLine.getAttribute("x1");
		curX = parseInt(curX);
		
		//convert direction to pixels (one value assumes moving on tick marks,
		//other values interpreted as absolute pixel moves). could add a toggle
		//option: TODO?
		var dx = direction;
		if (direction == 1 || direction == -1) {
			dx = this.rulerMarks.tickSpace * direction;
		}
		
		//find newX for slider line
		var newX = curX + dx;
		
		//boundary checks so doesn't move too far
		if ( newX <= this.rulerMarks.tickExes[0] && direction < 0) {
			newX = this.rulerMarks.tickExes[0];
			dx = newX - curX;
		} else if (newX >= this.rulerMarks.tickExes[this.rulerMarks.tickExes.length - 1] && direction > 0 ) {
			newX = this.rulerMarks.tickExes[this.rulerMarks.tickExes.length - 1];
			dx = newX - curX;
		}
		
		//move the slider line
		this.sliderControl.sliderControlLine.setAttribute("x1", newX);
		this.sliderControl.sliderControlLine.setAttribute("x2", newX);
		//move the slider background
		curX = this.sliderControl.sliderControlBG.getAttribute("x");
		newX = parseInt(curX) + dx;
		this.sliderControl.sliderControlBG.setAttribute("x", newX);
		//move the slider rect
		curX = this.sliderControl.sliderControl.getAttribute("x");
		newX = parseInt(curX) + dx;
		this.sliderControl.sliderControl.setAttribute("x", newX);
	};

	Slider.prototype.snapToClosest = function () {
		var sliderControlX = parseInt(this.sliderControl.sliderControlLine.getAttribute("x1"));
		var snapIndex = 0;// = slider.rulerMarks.curIndex;
		var minDist = 10000; //starter val too big for itself
		var dist = 10000;
		for (i = 0; i < this.rulerMarks.tickExes.length; i++) {
			dist = this.rulerMarks.tickExes[i] - sliderControlX;
			if ( Math.abs(dist) < Math.abs(minDist) ) {
				minDist = dist;
				snapIndex = i;
			}
		}
		this.moveSlider(minDist);
	};

	//user fills in these prototype functions below to state what should happen
	//in addition to the built-in event handlers
	//the built-in event handlers (arrowDownHandler, arrowUpHandler, etc)
	//only move the slider controls but the user will want these controls
	//to be modifying other elements when they change, such as colors inside rectangle elements
	//in the document.
	Slider.prototype.onArrowDown; //for mouse down/held-down on arrow button of slider
	Slider.prototype.onSliderDrag; //for mouse move when sliderControl being dragged
	Slider.prototype.onArrowUp; //for mouse lifting up or leaving an arrow button of slider

//Addition Demo Stuff
	function AdditionDemo(canv, placement1, placement2, circRadius) {
		this.can = canv;
		this.ctx = this.can.getContext("2d");
		
		//var to track which circle is being dragged
		this.dragCirc;
		
		//used for avoiding jumps on mousedowns prior to dragging
		//will track offset from mouse clicks to center of current
		//circle that has been clicked
		this.distToCenterX = 0;
		this.distToCenterY = 0;
		
		//vars for boundaries of draggable area
		//let them go off edge a little bit
		this.minX = 0;
		this.maxX = this.can.width;
		this.minY = 0;
		this.maxY = this.can.height;
		
		//Set overlapping treatment to addition
		this.ctx.globalCompositeOperation = "lighter";
		
		//Create circles and draw them to canvas
		this.rCirc = new Circle(placement1, placement1, circRadius, "#FF0000");
		this.gCirc = new Circle(placement2, placement1, circRadius, "#00FF00");
		this.bCirc = new Circle(placement2, placement2, circRadius, "#0000FF");
		this.circs = [this.rCirc, this.gCirc, this.bCirc];
		this.refreshCanvas();
		this.addEventListeners();
		
		//Drag moves a circle by a tiny bit based on mouse position
		//will be called on every mousemove when a circle is being
		//dragged. Also refreshes the canvas.
		var self = this;
		this.drag = function () {
			//figure out where circle should move
			var curX = self.dragCirc.x + self.distToCenterX;
			var curY = self.dragCirc.y + self.distToCenterY;
			var clickPt = self.mouseCanPos(event.clientX, event.clientY);
			var dx = clickPt.x - curX;
			var dy = clickPt.y - curY;
			self.dragCirc.move(dx, dy);
			//if circle went out of bounds move it back in
			if (self.dragCirc.x < self.minX || self.dragCirc.x > self.maxX
				|| self.dragCirc.y < self.minY || self.dragCirc.y > self.maxY) {
				self.dragCirc.move(-dx, -dy);
			}
			self.refreshCanvas();		
		};

		//To add mousemove and mouseup event listeners to window
		//when a drag has been initiated on a circle
		this.attachDrag = function () {
			window.addEventListener("mousemove", self.drag);
			window.addEventListener("mouseup", function () {
				window.removeEventListener("mousemove", self.drag);
				self.distToCenterX = 0;
				self.distToCenterY = 0;
			});
		};
	}

	AdditionDemo.prototype.addEventListeners = function () {
		//Add event listener for mousedown events on the canvas
		//check to see if click occurred within boundaries of a color
		//circle object. If so, mark this circle as clicked and call
		//the attachDrag function to set up mousemove and mouseup event
		//listeners for the drag operation on the window
		var self = this;
		this.can.addEventListener("mousedown", function () {
			var clickPt = self.mouseCanPos(event.clientX, event.clientY);
			if ( ptInCircle(self.rCirc, clickPt.x, clickPt.y) ) {
				self.distToCenterX = clickPt.x - self.rCirc.x;
				self.distToCenterY = clickPt.y - self.rCirc.y;
				self.dragCirc = self.rCirc;
				self.attachDrag();
			} else if ( ptInCircle(self.gCirc, clickPt.x, clickPt.y) ) {
				self.distToCenterX = clickPt.x - self.gCirc.x;
				self.distToCenterY = clickPt.y - self.gCirc.y;
				self.dragCirc = self.gCirc;
				self.attachDrag();
			} else if ( ptInCircle(self.bCirc, clickPt.x, clickPt.y) ) {
				self.distToCenterX = clickPt.x - self.bCirc.x;
				self.distToCenterY = clickPt.y - self.bCirc.y;
				self.dragCirc = self.bCirc;
				self.attachDrag();
			}
		});
	};
		
	//Draws current state of our circles to the canvas
	AdditionDemo.prototype.refreshCanvas = function() {
		this.ctx.clearRect(0, 0, this.can.width, this.can.height);
		for (var i = 0; i < this.circs.length; i++) {
			drawCircle(this.ctx, this.circs[i]);
		}
	};
		
	//Given an x and y in mouse coordinates, convert to canvas
	//coordinates
	AdditionDemo.prototype.mouseCanPos = function(xVal, yVal) {
		var rect = this.can.getBoundingClientRect();
		return {
			x: xVal - rect.left,
			y: yVal - rect.top
		};
	};
	
	//Function to reset the circle colors of the addition demo
	//to new colors (as hex strs)
	AdditionDemo.prototype.updateCircleColors = function(rVal, gVal, bVal) {
		this.rCirc.color = rVal;
		this.gCirc.color = gVal;
		this.bCirc.color = bVal;
		this.refreshCanvas();
	};
		
	//Circle constructor, data members x and y coordinates,
	//radius, and color as a hex-value string
	function Circle (xVal, yVal, radVal, col) {
		this.x = xVal;
		this.y = yVal;
		this.radius = radVal;
		this.color = col;
	}

	//Method to "move" circle, updates x and y data members
	//based on dx, dy
	Circle.prototype.move = function (dx, dy) {
		this.x = this.x + dx;
		this.y = this.y + dy;
	};
		
	//Function to draw a circle given a canvas context and
	//a reference to the circle to be drawn
	function drawCircle(context, circ) {
		context.beginPath();
		context.strokeStyle = "#FFFFFF";
		context.fillStyle = circ.color;
		context.arc(circ.x, circ.y, circ.radius, 0, Math.PI*2);
		context.stroke();
		context.fill();
	}

	//Determines if a given x, y point is within the boundaries
	//of a given circle object
	function ptInCircle(circ, x, y) {
		var dist = Math.sqrt( Math.pow(circ.x - x, 2) + Math.pow(circ.y - y, 2) );
		if (dist < circ.radius) {
			return true;
		}
		else {
			return false;
		}
	}	

//Subtraction Demo Stuff
	function SubtractionDemo (canv, placement1, placement2, w, h) {
		this.can = canv;
		this.ctx = this.can.getContext("2d");
		
		//var to track which rectangle is being dragged
		this.dragRect;
		
		//used for avoiding jumps on mousedowns prior to dragging
		//will track offset from mouse clicks to center of current
		//rectangle that has been clicked
		this.distToCenterX = 0;
		this.distToCenterY = 0;
		
		//vars for boundaries of draggable area
		//let them go off edge a little bit
		this.minX = -w*(3.0/4);
		this.maxX = this.can.width - w/2.0;
		this.minY = -h*(3.0/4);
		this.maxY = this.can.height - w/2.0;
		
		//Set overlapping treatment to subtraction
		this.ctx.globalCompositeOperation = "multiply";
		
		//Create rectangles and draw them to canvas
		this.cRect = new Rectangle(placement1, placement1, w, h, "#00FFFF");
		this.mRect = new Rectangle(placement2, placement1, w, h, "#FF00FF");
		this.yRect = new Rectangle(placement2, placement2, w, h, "#FFFF00");
		this.rects = [this.cRect, this.mRect, this.yRect];
		this.refreshCanvas();
		this.addEventListeners();
		
		var self = this;
		//Drag moves a rectangle by a tiny bit based on mouse position
		//will be called on every mousemove when a rectangle is being
		//dragged. Also refreshes the canvas.
		this.drag = function () {
			//figure out where rectangle should move
			var curX = self.dragRect.x + self.distToCenterX;
			var curY = self.dragRect.y + self.distToCenterY;
			var clickPt = self.mouseCanPos(event.clientX, event.clientY);
			var dx = clickPt.x - curX;
			var dy = clickPt.y - curY;
			self.dragRect.move(dx, dy);
			//if rectangle went out of bounds move it back in
			if (self.dragRect.x < self.minX || self.dragRect.x > self.maxX
				|| self.dragRect.y < self.minY || self.dragRect.y > self.maxY) {
				self.dragRect.move(-dx, -dy);
			}
			self.refreshCanvas();		
		};
		
		//To add mousemove and mouseup event listeners to window
		//when a drag has been initiated on a rectangle
		this.attachDrag = function() {
			window.addEventListener("mousemove", self.drag);
			window.addEventListener("mouseup", function () {
				window.removeEventListener("mousemove", self.drag);
				self.distToCenterX = 0;
				self.distToCenterY = 0;
			});
		};
	}

	//Add event listener for mousedown events on the canvas
	//check to see if click occurred within boundaries of aLinkcolor
	//rectangle object. If so, mark this rectangle as clicked and call
	//the attachDrag function to set up mousemove and mouseup event
	//listeners for the drag operation on the window
	SubtractionDemo.prototype.addEventListeners = function () {
		var self = this;
		this.can.addEventListener("mousedown", function () {
			var clickPt = self.mouseCanPos(event.clientX, event.clientY);
			if ( ptInRectangle(self.cRect, clickPt.x, clickPt.y) ) {
				self.distToCenterX = clickPt.x - self.cRect.x;
				self.distToCenterY = clickPt.y - self.cRect.y;
				self.dragRect = self.cRect;
				self.attachDrag();
			} else if ( ptInRectangle(self.mRect, clickPt.x, clickPt.y) ) {
				self.distToCenterX = clickPt.x - self.mRect.x;
				self.distToCenterY = clickPt.y - self.mRect.y;
				self.dragRect = self.mRect;
				self.attachDrag();
			} else if ( ptInRectangle(self.yRect, clickPt.x, clickPt.y) ) {
				self.distToCenterX = clickPt.x - self.yRect.x;
				self.distToCenterY = clickPt.y - self.yRect.y;
				self.dragRect = self.yRect;
				self.attachDrag();
			}
		});
	};
	
	//Draws current state of our rectangles to the canvas
	SubtractionDemo.prototype.refreshCanvas = function () {
		this.ctx.clearRect(0, 0, this.can.width, this.can.height);
		for (var i = 0; i < this.rects.length; i++) {
			drawRectangle(this.ctx, this.rects[i]);
		}
	};
	
	//Given an x and y in mouse coordinates, convert to canvas
	//coordinates
	SubtractionDemo.prototype.mouseCanPos = function(xVal, yVal) {
		var rect = this.can.getBoundingClientRect();
		return {
			x: xVal - rect.left,
			y: yVal - rect.top
		};
	};

	//Function to reset the rect colors of the subtraction demo
	//to new colors (as hex strs)
	SubtractionDemo.prototype.updateRectColors = function(cVal, mVal, yVal) {
		this.cRect.color = cVal;
		this.mRect.color = mVal;
		this.yRect.color = yVal;
		this.refreshCanvas();
	};
	
	//rectangle constructor, data members x and y coordinates
	//of top left, width, height, and color as a hex-value string
	function Rectangle (xVal, yVal, widthVal, heightVal, col) {
		this.x = xVal;
		this.y = yVal;
		this.width = widthVal;
		this.height = heightVal;
		this.color = col;
	}

	//Method to "move" rectangle, updates x and y data members
	//based on dx, dy
	Rectangle.prototype.move = function (dx, dy) {
		this.x = this.x + dx;
		this.y = this.y + dy;
	};
		
	//Function to draw a rectangle given a canvas context and
	//a reference to the rectangle to be drawn
	function drawRectangle(context, rect) {
		context.strokeStyle = "#000000";
		context.lineWidth=1.5;
		context.fillStyle = rect.color;
		context.strokeRect(rect.x, rect.y, rect.width, rect.height);
		context.fillRect(rect.x, rect.y, rect.width, rect.height);
	}

	//Determines if a given x, y point is within the boundaries
	//of a given rectangle object
	function ptInRectangle(rect, x, y) {
		var minX = rect.x;
		var maxX = rect.x + rect.width;
		var minY = rect.y;
		var maxY = rect.y + rect.height;
		if ( x >= minX && x <= maxX &&
			 y >= minY && y <= maxY) {
			return true;
		} else {
			return false;
		}
	}
	
//The Return	
	return {
		"Slider" : Slider,
		"AdditionDemo" : AdditionDemo,
		"SubtractionDemo" : SubtractionDemo
	}
}();
