/*
+---------------------------------------------------------+
| Frances Ruiz and Michael J. Ruiz                        |
+---------------------------------------------------------|
| License:                                                |
| Creative Commons Attribution-NonCommerial 4.0 Unported  |
| http://creativecommons.org/licenses/by-nc/4.0/legalcode |
+---------------------------------------------------------+
| You are free to:                                        |
|                                                         |
| Share  copy and redistribute the material in any       |
|         medium or format                                |
| Adapt  remix, transform, and build upon the material   |
|                                                         |
| The licensor cannot revoke these freedoms as long as    |
| you follow the license terms.                           |
|                                                         |
| Attribution  You must give appropriate credit, include |
| this license, and indicate if changes were made. You    |
| may do so in any reasonable manner, but not in any way  |
| that suggests the licensor endorses you or your use.    |
|                                                         |
| Sample Credit Line: Courtesy Frances and Michael Ruiz   |
|                                                         |
| NonCommercial  You may not use the material for        |
|   commercial purposes.                                  |
|                                                         |
| No additional restrictions  You may not apply legal    |
| terms or technological measures that legally restrict   |
| others from doing anything the license permits.         |
+---------------------------------------------------------+
*/

function init() {
	//bring in canvases, set up contexts
	var can = document.getElementById("pic_can");
	var redCan = document.getElementById("red_can");
	var blueCan = document.getElementById("blue_can");
	var greenCan = document.getElementById("green_can");
	var ctx = can.getContext("2d");
	var redCtx = redCan.getContext("2d");
	var blueCtx = blueCan.getContext("2d");
	var greenCtx = greenCan.getContext("2d");

	//Create red/green/blue Pictures starting with the orig_pic
	//as the base, and place in their respective contexts.
	var origPicElement=document.getElementById("orig_pic");
	var origPicWidth = parseInt(origPicElement.getAttribute("width"));
	var origPicHeight = parseInt(origPicElement.getAttribute("height"));
	var redCanPic = new Picture(origPicElement, redCtx, 0, 0, origPicWidth, origPicHeight);
	redCanPic.toRed();
	var greenCanPic = new Picture(origPicElement, greenCtx, 0, 0, origPicWidth, origPicHeight);
	greenCanPic.toGreen();
	var blueCanPic = new Picture(origPicElement, blueCtx, 0, 0, origPicWidth, origPicHeight);
	blueCanPic.toBlue();
	
	//to keep track of which Rectangle is being dragged
	var dragRect;
	
	//Set overlapping treatment to addition
	ctx.globalCompositeOperation = "lighter";
	
	//used for avoiding jumps on mousedowns prior to dragging
	//will track offset from mouse clicks to center of current
	//Picture that has been clicked
	var distToCenterX = 0;
	var distToCenterY = 0;

	//Make rectangle objects that will contain the rgb images
	var rRect = new Rectangle(0, 0, origPicWidth, origPicHeight)
	var gRect = new Rectangle(origPicWidth + 10, 0, origPicWidth, origPicHeight)
	var bRect = new Rectangle(0, origPicHeight + 10, origPicWidth, origPicHeight)
	var rects = [rRect, gRect, bRect];
	refreshCanvas();
	
	//Given an x and y in mouse coordinates, convert to canvas
	//coordinates
	function mouseCanPos(xVal, yVal) {
		var rect = can.getBoundingClientRect();
		return {
			x: xVal - rect.left,
			y: yVal - rect.top
		};
	}

	//Add event listener for mousedown events on the canvas
	//check to see if click occurred within boundaries of a Picture
	//object. If so, mark this Picture as clicked and call
	//the attachDrag function to set up mousemove and mouseup event
	//listeners for the drag operation on the window
	can.addEventListener("mousedown", function () {
		var clickPt = mouseCanPos(event.clientX, event.clientY);
		for (var i = 0; i < rects.length; i++) { //pics.length; i++) {
			if (ptInRectangle(rects[i], clickPt.x, clickPt.y)) {
				distToCenterX = clickPt.x - rects[i].x;
				distToCenterY = clickPt.y - rects[i].y;
				dragRect = rects[i];
				attachDrag();
				break; //layering so first pic always "on top"
			}
		}
	});
	
	//To add mousemove and mouseup event listeners to window
	//when a drag has been initiated on a rectangle
	function attachDrag() {
		window.addEventListener("mousemove", drag);
		window.addEventListener("mouseup", function () {
			window.removeEventListener("mousemove", drag);
			distToCenterX = 0;
			distToCenterY = 0;
			if (outOfBounds(dragRect)) {
				moveInBounds(dragRect);
				refreshCanvas();
			}
		});
	}
	
	//Tests is a rectangle is out of bounds, as
	//defined by the main canvas and the width/height
	//of the rectangle
	function outOfBounds(rect) {
		var xBound = can.width - rect.width;
		var yBound = can.height - rect.height;
		if (rect.x < 0 || rect.x > xBound )
			return true;
		else if (rect.y < 0 || rect.y > yBound)
			return true;
		else
			return false;
	}
	
	//Moves a rectangle back in bounds
	function moveInBounds(rect) {
		var xBound = can.width - rect.width;
		var yBound = can.height - rect.height;
		if (rect.x < 0)
			rect.move(Math.abs(rect.x), 0);
		if (rect.y < 0)
			rect.move(0, Math.abs(rect.y));
		if (rect.x > xBound)
			rect.move( -1 * (rect.x - xBound), 0);
		if (rect.y > yBound)
			rect.move(0, -1 * (rect.y - yBound));
	}
	
	//Draws current state of our rectangles to the canvas
	function refreshCanvas () {
		ctx.clearRect(0, 0, can.width, can.height);
		copyRectangle(redCan, ctx, rRect);
		copyRectangle(blueCan, ctx, bRect);
		copyRectangle(greenCan, ctx, gRect);
	}

	//Drag moves a rectangle by a tiny bit based on mouse position
	//will be called on every mousemove when a rectangle is being
	//dragged. Also refreshes the canvas.
	function drag () {
		//figure out where rectangle should move
		var curX = dragRect.x + distToCenterX;
		var curY = dragRect.y + distToCenterY;
		var clickPt = mouseCanPos(event.clientX, event.clientY);
		var dx = clickPt.x - curX;
		var dy = clickPt.y - curY;
		dragRect.move(dx, dy);
		//if rectangle went out of bounds move it back in
		if (outOfBounds(dragRect))
			moveInBounds(dragRect);
		refreshCanvas();		
	}
}
	
//rectangle constructor, data members x and y coordinates
//of top left, width, height
function Rectangle (xVal, yVal, widthVal, heightVal) {
	this.x = xVal;
	this.y = yVal;
	this.width = widthVal;
	this.height = heightVal;
}

//Method to "move" rectangle, updates x and y data members
//based on dx, dy
Rectangle.prototype.move = function (dx, dy) {
	this.x = this.x + dx;
	this.y = this.y + dy;
};
	
//Function to "copy" a rectangle onto a context given
//the source canvas and dimensions (assumes source canvas
//matches size of rect)
function copyRectangle(canvas, context, rect) {
	context.drawImage(canvas, 0, 0, canvas.width, canvas.height,
					  rect.x, rect.y , rect.width, rect.height);
}

//Determines if a given x, y point is within the boundaries
//of a given rectangle object
function ptInRectangle(rect, x, y) {
	var minX = rect.x;
	var maxX = rect.x + rect.width;
	var minY = rect.y;
	var maxY = rect.y + rect.height;
	if ( x >= minX && x <= maxX &&
		 y >= minY && y <= maxY) {
		return true;
	} else {
		return false;
	}
}

//Constructor for a Picture object which will need an
//image element, a context2d, an x and y value, and width
//and height values. Picture objs represent/keep track of a
//picture on a canvas and can manipulate its colors.
function Picture(imgElement, context, x, y, width, height) {
	context.drawImage(imgElement, x, y, width, height);
	this.pic = imgElement;
	this.ctx = context;
	this.x = x;
	this.y = y;
	this.width = width;
	this.height = height;
	this.imgData = this.ctx.getImageData(this.x, this.y, this.width, this.height);
}

//Convert image to red by setting g and b values to zero
//for every pixel but leaving r values alone
Picture.prototype.toRed = function() {
	for (var i = 0; i < this.imgData.data.length; i+=4)
	{
		this.imgData.data[i+1]=0;
		this.imgData.data[i+2]=0;	
	}
	this.ctx.putImageData(this.imgData, this.x, this.y);
};

//Convert image to blue by setting g and r values to zero
//for every pixel but leaving b values alone
Picture.prototype.toBlue = function() {
	for (var i = 0; i < this.imgData.data.length; i+=4)
	{
		this.imgData.data[i]=0;
		this.imgData.data[i+1]=0;	
	}
	this.ctx.putImageData(this.imgData, this.x, this.y);
};

//Convert image to green by setting b and r values to zero
//for every pixel but leaving g values alone
Picture.prototype.toGreen = function() {
	for (var i = 0; i < this.imgData.data.length; i+=4)
	{
		this.imgData.data[i]=0;
		this.imgData.data[i+2]=0;	
	}
	this.ctx.putImageData(this.imgData, this.x, this.y);
};

//Convert image to cyan by setting g and b values to 255
//for every pixel but leaving r values alone
Picture.prototype.toCyan = function() {
	for (var i = 0; i < this.imgData.data.length; i+=4)
	{
		this.imgData.data[i+1]=255;
		this.imgData.data[i+2]=255;	
	}
	this.ctx.putImageData(this.imgData, this.x, this.y);
};

//Convert image to magenta by setting b and r values to zero
//for every pixel but leaving g values alone
Picture.prototype.toMagenta = function() {
	for (var i = 0; i < this.imgData.data.length; i+=4)
	{
		this.imgData.data[i]=255;
		this.imgData.data[i+2]=255;	
	}
	this.ctx.putImageData(this.imgData, this.x, this.y);
};

//Convert image to yellow by setting g and r values to 255
//for every pixel but leaving b values alone
Picture.prototype.toYellow = function() {
	for (var i = 0; i < this.imgData.data.length; i+=4)
	{
		this.imgData.data[i]=255;
		this.imgData.data[i+1]=255;
	}
	this.ctx.putImageData(this.imgData, this.x, this.y);
};

window.onload = init;
